# copyright 2015

#    under the License.

"""
Backup driver agent for huawei eBackup

"""
import functools
import json
import random
import re
import socket
import ssl
import sys
import threading
import time
import urllib3
import uuid
from ctypes import CDLL, c_int, c_char_p
import subprocess
import os
import requests
from oslo_utils.uuidutils import is_uuid_like

from cinder.backup.drivers.ebackupconst import Constants as cst
from cinder.backup.drivers.ebackupconst import ErrorCode

import six
if six.PY3:
    import http.client as http_client
    from urllib.error import HTTPError, URLError
    from urllib.request import HTTPSHandler, urlopen
    from urllib.request import ProxyHandler, build_opener, Request
else:
    import httplib as http_client
    from urllib2 import HTTPError, URLError, HTTPSHandler, urlopen
    from urllib2 import ProxyHandler, build_opener, Request


try:
    # M version
    from oslo_config import cfg
    from oslo_log import log as logging
    from oslo_utils import excutils
    try:
        from oslo_utils import strutils
        strutils._SANITIZE_KEYS.append('ak')
        strutils._SANITIZE_KEYS.append('sk')
        for key in strutils._SANITIZE_KEYS:
            strutils._SANITIZE_PATTERNS_1[key] = []
            strutils._SANITIZE_PATTERNS_2[key] = []
            for pattern in strutils._FORMAT_PATTERNS_2:
                reg_ex = re.compile(pattern % {'key': key}, re.DOTALL)
                strutils._SANITIZE_PATTERNS_2[key].append(reg_ex)
            for pattern in strutils._FORMAT_PATTERNS_1:
                reg_ex = re.compile(pattern % {'key': key}, re.DOTALL)
                strutils._SANITIZE_PATTERNS_1[key].append(reg_ex)
    except Exception as e:
        logging.getLogger(__name__).info("Try hidding aksk with excutils failed! %s" % e)
except:
    # J version
    from oslo.config import cfg
    from cinder.openstack.common import log as logging
    from cinder.openstack.common import excutils

from oslo_utils import importutils

sessionmap = None
osprofiler_web = importutils.try_import('osprofiler.web')

LOG = logging.getLogger(__name__)
# file path
file_abs_path = os.path.dirname(os.path.abspath(__file__))
_KMC_LIB_PATH = str(file_abs_path) + "/../lib/libkmcadaptor.so"

# check if env is DJ
DJ_ENV_FILE = '/etc/huawei/dj/cfg/sys.ini'
g_in_ebackupconfig = False


def is_dj_env():
    if os.path.exists(DJ_ENV_FILE):
        return True
    return False
try:
    try:
        from FSSecurity import crypt as fscrypt
    except Exception:
        from FSComponentUtil import crypt as fscrypt
except Exception as msg:
    is_dj_env = is_dj_env()
    if is_dj_env == False:
        raise Exception(msg)

hws_opts = [
    cfg.StrOpt('ebackup_task_retry_count',
               default='0',
               help=''),
    cfg.StrOpt('ebackup_task_retry_time_interval',
               default='300',
               help=''),
    cfg.StrOpt('api_gateway_host',
               default='',
               help=''),
    cfg.StrOpt('api_gateway_ca_file',
               default='',
               help=''),
    cfg.StrOpt('sync_pools',
               default='90',
               help=''),
    cfg.StrOpt('async_pools',
               default='10',
               help='')
]

if sys.version_info >= (2, 7, 5):
    ssl._create_default_https_context = ssl._create_unverified_context

CINDER_CONF = cfg.CONF
CINDER_CONF.register_opts(hws_opts)

sync_pool = urllib3.PoolManager(cert_reqs='CERT_NONE', num_pools=int(CINDER_CONF.sync_pools), block=True)
async_pool = urllib3.PoolManager(cert_reqs='CERT_NONE', num_pools=int(CINDER_CONF.async_pools), block=True)


def convert_error(error_code, error_desc="eBackup return error", error_params=None):
    error_json = dict()
    error_json["code"] = str(error_code)
    error_json["desc"] = error_desc
    if not error_params:
        # error_params format should be ["param1", "param2"]
        error_json["params"] = error_params
    error_ret = json.dumps(error_json)
    return error_ret


class Utils(object):
    """ eBackup driver utils function"""

    @staticmethod
    def decrypt_password(encrypt_pwd):
        ''' decrypt password by iv'''
        if is_dj_env():  # if  at  dj env  , use kmc
            KMC = _load_kmc()
            if KMC is not None:  # if kmc is not null
                return KMC.aesDecrypt(0, encrypt_pwd)
            return encrypt_pwd
        # if not dj env , return
        try:
            de_pwd = fscrypt.decrypt(encrypt_pwd)  # for ebackupconfig
            return de_pwd
        except Exception as msg:
            LOG.debug(msg)
            return encrypt_pwd

    @staticmethod
    def encrypt_password(decrypt):
        ''' encrypt password by iv'''
        try:
            en_pwd = fscrypt.encrypt(decrypt)
            return en_pwd
        except Exception as msg:
            LOG.error(msg)

    @staticmethod
    def check_string(src_string, compile_re):
        ''' match the src string by the regular'''
        re_com = re.compile(compile_re)
        if re_com.match(src_string):
            return True
        else:
            return False

    @staticmethod
    def get_hostname():
        try:
            with open("/etc/uuid") as f:
                return f.readline().rstrip("\n")
        except Exception as msg:
            LOG.error(msg)
            return ''

    @staticmethod
    def transfer_to_json_str(json_dict):
        result = ""
        try:
            if not isinstance(json_dict, dict):
                LOG.info("body isn't dictionary")
                return result
            result = json.dumps(json_dict, separators=(',', ":"))
        except Exception as e:
            LOG.info("exception:%s" % e)
        return result

    @staticmethod
    def transfer_json_to_dict(src):
        result = {}
        try:
            result = json.loads(src)
            if not isinstance(result, dict):
                result = {}
        except Exception as e:
            LOG.info("str is not json object:%s" % e)
        return result

    @staticmethod
    def gen_request_id():
        req_id = "%d" % hash("%s" % uuid.uuid1())
        if req_id.startswith('-'):
            req_id = req_id[1:]
        return req_id

    @staticmethod
    def get_fsp_cascaded_om_ip():
        child1 = subprocess.Popen(["ip", "a"], shell=False,
                                  stdout=subprocess.PIPE)
        child2 = subprocess.Popen(["grep", "-i", "global external_om"],
                                  stdin=child1.stdout, stdout=subprocess.PIPE,
                                  shell=False)
        output = child2.communicate()[0]
        if six.PY3:
            output = str(output, encoding='utf-8')
        if len(output) == 0:
            LOG.error("Get FSP om ip failed")
            return ""
        else:
            tmptxt = output.split()
            # tmptxt:['inet', 'x.x.x.x/x', 'scope', 'global', 'external_om']
            om_ip = tmptxt[1].split('/')
            return om_ip[0]


class BoundHTTPsHandler(HTTPSHandler):
    def __init__(self, source_address=None, debuglevel=0):
        HTTPSHandler.__init__(self, debuglevel)
        self.https_class = functools.partial(http_client.HTTPSConnection,
                                             source_address=source_address)

    def https_open(self, req):
        return self.do_open(self.https_class, req)


def call_db_fun(db_fun, *args, **kwargs):
    errmsg = 'call db success'
    for retry_time in range(0, cst.DB_ERROR_RETRY_TIMES):
        try:
            LOG.debug("call db fun:%s." % db_fun.__name__)
            return db_fun(*args, **kwargs)
        except Exception as msg:
            LOG.error("call db fun:%s failed:%s" % (db_fun.__name__, msg))
            time.sleep(cst.DB_ERROR_RETRY_INTERVAL)
            continue
    raise Exception(errmsg)


class SessionMap(object):
    def __init__(self, session=None, lock=None):
        if session:
            self.session = session
        else:
            self.session = None
        if lock:
            self.checklock = lock
        else:
            self.checklock = threading.Lock()


class HttpConnection(object):
    """ http restful for eBackup"""

    def __init__(self, ebackup_conf):
        '''
        __init__

        :return:
        '''
        self.body = None
        self.doing_login = False
        self.login_param = None
        self.checklock = threading.Lock()
        self.ebackup_conf = ebackup_conf
        global sessionmap
        if not sessionmap:
            LOG.info('init global variable session map')
            sessionmap = dict()
        if not sessionmap.get(self.ebackup_conf.ebackup_server_ip):
            sessionmap[self.ebackup_conf.ebackup_server_ip] = SessionMap()
        try:
            re_com = re.compile(cst.ip_v6_compile)
            if re_com.match(self.ebackup_conf.ebackup_server_ip):
                self.cascaded_om_ip = ""
                self.ebackup_conf.ebackup_server_ip = '[' + self.ebackup_conf.ebackup_server_ip + ']'
            else:
                self.cascaded_om_ip = Utils.get_fsp_cascaded_om_ip()
        except Exception as msg:
            LOG.error('Connect eBackup failed, msg : %s.' % msg)
            self.cascaded_om_ip = ""

    def get_phy_network_type(self):
        if self.ebackup_conf.physical_network_type != '':
            return self.ebackup_conf.physical_network_type

        return 'ip'

    def get_fusion_storage_ip(self):
        return self.ebackup_conf.fusion_storage_ip

    def get_fusion_storage_agent_ip(self):
        return self.ebackup_conf.fusion_storage_agent_ip

    def get_ip(self):
        return self.ebackup_conf.ebackup_server_ip

    def get_port(self):
        return self.ebackup_conf.ebackup_server_port

    def _set_om_ip(self):
        try:
            regex = re.compile(cst.ip_v6_compile)
            if not regex.match(self.ebackup_conf.ebackup_server_ip):
                self.cascaded_om_ip = Utils.get_fsp_cascaded_om_ip()
        except Exception as ex:
            LOG.info('Failed to get om ip: {}.'.format(ex))

    def __getsession__(self, retry, force=False, old_session=None):
        if force or not sessionmap[self.ebackup_conf.ebackup_server_ip].session:
            LOG.info('get session lock of %s.' % self.ebackup_conf.ebackup_server_ip)
            sessionmap[self.ebackup_conf.ebackup_server_ip].checklock.acquire()
            if (not old_session) and sessionmap[self.ebackup_conf.ebackup_server_ip].session:
                LOG.info('Some other thread update the session of %s' % self.ebackup_conf.ebackup_server_ip)
                sessionmap[self.ebackup_conf.ebackup_server_ip].checklock.release()
                return sessionmap[self.ebackup_conf.ebackup_server_ip].session
            if old_session:
                if old_session.get('login_cookie') != sessionmap[self.ebackup_conf.ebackup_server_ip].session.get(
                        'login_cookie'):
                    LOG.info('Some other thread update the old session of %s' % self.ebackup_conf.ebackup_server_ip)
                    sessionmap[self.ebackup_conf.ebackup_server_ip].checklock.release()
                    return sessionmap[self.ebackup_conf.ebackup_server_ip].session
            LOG.info('login the ebackup server %s' % self.ebackup_conf.ebackup_server_ip)
            try:
                token = self.__login(retry)
                LOG.info('update session.')
                sessionmap[self.ebackup_conf.ebackup_server_ip].session = token
            finally:
                LOG.info('release session lock of %s.' % self.ebackup_conf.ebackup_server_ip)
                sessionmap[self.ebackup_conf.ebackup_server_ip].checklock.release()
            return sessionmap[self.ebackup_conf.ebackup_server_ip].session
        else:
            return sessionmap[self.ebackup_conf.ebackup_server_ip].session

    def send(self, filter_url, method, retry, body=None, sync=False):
        ''' get restful for eBackup'''
        cur_session = self.__getsession__(retry)
        conn_count = 1
        if filter_url.find("srv_jobmanager") < 0:
            url = cur_session['base_uri'] + str(filter_url)
        else:
            url = "https://" + self.ebackup_conf.ebackup_server_ip + ":" + self.ebackup_conf.ebackup_server_port + filter_url
        LOG.info(url)
        while True:
            try:
                header = dict()
                header['Content-Type'] = "application/json;charset=UTF-8"
                header['Accept-Language'] = "en"
                header['Accept'] = "application/json;version=2.3;charset=UTF-8"
                header['iBaseToken'] = cur_session['login_token']
                header['Cookie'] = cur_session['login_cookie']
                header['X-Auth-Token'] = cur_session['micro_service_token']
                if osprofiler_web:
                    trace_headers = osprofiler_web.get_trace_id_headers()
                    for key in trace_headers:
                        header[key] = trace_headers[key]
                if body:
                    encode_body = json.dumps(body, separators=(',', ':'), ensure_ascii=False).encode('utf-8')
                else:
                    encode_body = None
                if sync:
                    response = sync_pool.request(method, url, fields=None, headers=header, body=encode_body,
                                                 timeout=cst.TIME_SLEEP_90)
                else:
                    response = async_pool.request(method, url, fields=None, headers=header, body=encode_body,
                                                  timeout=cst.TIME_SLEEP_90)
                if response.status in [
                    cst.BAD_REQUEST,
                    cst.INTERBAL_SERVER_ERROR,
                    cst.SERVER_TIMEOUT,
                    cst.SERVICE_UNAVAILABLE,
                    cst.HTTP_BAD_GATEWAY,
                    cst.HTTP_GATEWAY_TIMEOUT
                ]:
                    err_msg = ('except Http error code, %s.' % str(response.status))
                    raise Exception(err_msg)
                result = json.loads(response.data)
                if result.get('Error', ''):
                    tmpError = result['Error']
                    tmpError['code'] = tmpError['Code']
                    tmpError['description'] = tmpError['Description']
                    result['error'] = tmpError
                if result.get('Data', ''):
                    result['data'] = result['Data']
                dict_error = result['error']
                error_code = dict_error['code']

                if error_code in [
                    ErrorCode.FAULT_LOGOUT, cst.EB_NO_AUTHENTICATION,
                    ErrorCode.INVALID_SESSION, cst.JM_NO_AUTHENTICATION,
                    ErrorCode.COMMUNICATION_FAILED, cst.EB_TOKEN_EXPIRED
                ]:
                    login_error_msg = "login eBackup failed, errorno is:" + str(error_code)
                    error_msg = convert_error(ErrorCode.BD_LOGIN_EBACKUP_FALIED, login_error_msg)
                    LOG.info(error_msg)
                    try:
                        cur_session = self.__getsession__(retry, True, cur_session)
                    except Exception:
                        with excutils.save_and_reraise_exception():
                            conn_count = cst.CONNECT_COUNT + 1
                            LOG.error("login retry count exceed 30, quit http req")
                    continue
                if error_code in ["-1", -1]:
                    err_msg = ('except ebackup error, %s.' % str(error_code))
                    raise Exception(err_msg)
                return result
            except Exception as msg:
                LOG.error(msg)
                if conn_count > cst.CONNECT_COUNT or not retry:
                    login_msg = ('Connect eBackup failed,%s.' % msg)
                    error_msg = convert_error(ErrorCode.BD_CONNECT_EBACKUP_FALIED, login_msg)
                    LOG.error(error_msg)
                    raise Exception(error_msg)
                else:
                    conn_count += 1
                    LOG.info('Connect eBackup failed, try again later.')
                    time.sleep(cst.TIME_SLEEP_300)
                    continue

    def __login(self, retry):
        ''' login ebackup '''
        ebackup_ip = self.ebackup_conf.ebackup_server_ip
        ebackup_port = self.ebackup_conf.ebackup_server_port
        port_compile = '^[0-9]*$'
        b_check_port = Utils.check_string(ebackup_port, port_compile)
        if not b_check_port:
            msg = "eBackup port in config file is not right, please check it."
            error_msg = convert_error(ErrorCode.BD_PORT_FORMAT_NOT_RIGHT, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)

        global g_in_ebackupconfig
        try:
            cacrtf = self.ebackup_conf.ebackup_login_crt
            if os.path.exists(cacrtf):
                ctx = ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
                ctx.verify_mode = ssl.CERT_REQUIRED
                ctx.check_hostname = False
                ctx.load_verify_locations(cacrtf)
                urlopen('https://' + ebackup_ip + ':' + ebackup_port, timeout=cst.TIME_SLEEP_60, context=ctx)
            else:
                if not g_in_ebackupconfig:
                    LOG.warn('certificate error: file is not exist.')
        except Exception as msg:
            if not g_in_ebackupconfig:
                LOG.warn('certificate error:' + str(msg))

        ebackup_login_uname = self.ebackup_conf.ebackup_login_username
        if ebackup_login_uname is None:
            msg = "eBackup username is Null."
            error_msg = convert_error(ErrorCode.EC_LOGIN_CRED_WRONG, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)

        login_pwd = self.ebackup_conf.ebackup_login_password
        if login_pwd is None:
            msg = "eBackup password is Null."
            error_msg = convert_error(ErrorCode.EC_LOGIN_CRED_WRONG, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)

        base_uri = 'https://' + ebackup_ip + ':' + ebackup_port + '/rest'
        login_uri = base_uri + '/dev/sessions'
        login_pwd = Utils.decrypt_password(login_pwd)
        if login_pwd == "":
            msg = "login password is wrong."
            error_msg = convert_error(ErrorCode.EC_LOGIN_CRED_WRONG, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)
        login_body = {
            "username": ebackup_login_uname,
            "password": login_pwd,
            "scope": cst.DOCKING_USER
        }
        login_pwd = None
        self.login_param = None
        retry_count = cst.RETRY_COUNT
        while True:
            try:
                proxy = ProxyHandler({})
                # get cascaded om ip
                if (self.cascaded_om_ip == ""):
                    opener = build_opener(proxy)
                else:
                    handler = BoundHTTPsHandler(source_address=(self.cascaded_om_ip, 0))
                    opener = build_opener(handler, proxy)
                jdata = json.dumps(login_body)
                if six.PY3:
                    jdata = jdata.encode('utf-8')
                req = Request(login_uri, jdata)
                response = opener.open(req, timeout=cst.TIME_SLEEP_60)
                if six.PY3:
                    full_cookie = response.getheader("Set-Cookie")
                else:
                    full_cookie = response.info().getheader("Set-Cookie")
                login_cookie = full_cookie[:full_cookie.find(';')]
                meta_data = json.loads(response.read())
                error_num = meta_data['error']['code']
                # the max session error code.
                elt = [ErrorCode.MAX_SESSIONS, ErrorCode.COMMUNICATION_FAILED]
                if error_num in elt:
                    if 0 == retry_count:
                        msg = 'connection failed.'
                        error_msg = convert_error(ErrorCode.BD_CONNECT_EBACKUP_FALIED, 'connect ebackup failed')
                        LOG.error(msg)
                        raise Exception(error_msg)
                    else:
                        retry_count -= 1
                        msg = 'The max session number, connection later.'
                        LOG.info(msg)
                        time.sleep(cst.TIME_SLEEP_60)
                        continue
                elif error_num != 0:
                    error_msg = convert_error(error_num, "login response data is null")
                    LOG.error(error_msg)
                    raise Exception(error_msg)
                login_token = meta_data.get('data').get("iBaseToken")
                deviceid = meta_data.get('data').get("deviceid")
                clientiptoken = meta_data.get('data').get("clientiptoken")
                base_uri += "/%s" % deviceid

                login_param = {'base_uri': base_uri,
                               'login_token': login_token,
                               'login_cookie': login_cookie,
                               'micro_service_token': clientiptoken}
                return login_param
            except Exception as msg:
                self._set_om_ip()
                retry_count -= 1
                LOG.exception(msg)
                login_msg = 'Login to ebackup failed, ' \
                            'ebackup ip:{ebk_ip}, ' \
                            'om ip:{om_ip}.'.format(ebk_ip=ebackup_ip,
                                                    om_ip=self.cascaded_om_ip)
                LOG.error(login_msg)
                error_msg = convert_error(ErrorCode.BD_CONNECT_EBACKUP_FALIED,
                                          "Failed to login ebackup.")
                if retry and retry_count > 0:
                    LOG.error('retry count %d' % retry_count)
                    time.sleep(cst.TIME_SLEEP_60)
                else:
                    LOG.error('retry completed!')
                    raise Exception(error_msg)


class Connection(HttpConnection):
    def __init__(self, ebackup_conf):
        super(Connection, self).__init__(ebackup_conf)

    def retry(error_should_retry):
        def func_wrapper(func):
            @functools.wraps(func)
            def real_wrapper(*argv, **kwargs):
                count = 0
                while count <= cst.BUSINESS_RETRY_COUNT:
                    result = func(*argv, **kwargs)
                    error_code = result["error"]["code"]
                    if error_code not in error_should_retry:
                        break
                    LOG.info("ebackup return error code[%s], retry:%d" % (error_code, count))
                    time.sleep(cst.TIME_SLEEP_300)
                    count += 1
                return result

            return real_wrapper

        return func_wrapper

    @retry([ErrorCode.EBK_IBASE_TIMEOUT, ErrorCode.EC_COM_SYSTEM_BUSY])
    def get(self, filter, retryed=True, sync=False):
        return super(Connection, self).send(filter, 'GET', retryed, None, sync)

    @retry([ErrorCode.EBK_IBASE_TIMEOUT, ErrorCode.EC_COM_SYSTEM_BUSY])
    def put(self, filter, body, retryed=True, sync=False):
        return super(Connection, self).send(filter, 'PUT', retryed, body, sync)

    @retry([ErrorCode.EBK_IBASE_TIMEOUT, ErrorCode.EC_COM_SYSTEM_BUSY])
    def post(self, filter, values, retryed=True, sync=False):
        return super(Connection, self).send(filter, 'POST', retryed, values, sync)

    @retry([ErrorCode.EBK_IBASE_TIMEOUT, ErrorCode.EC_COM_SYSTEM_BUSY])
    def delete(self, filter, body=None, retryed=True, sync=False):
        return super(Connection, self).send(filter, 'DELETE', retryed, body, sync)


class RestConnection(object):  # http try three times
    def __init__(self):
        pass

    def get(self, url=None, headers=None):
        pass

    def post(self, url=None, headers=None, body=None, context=None):
        LOG.info("URL:%s", url)
        conn_count = 1
        while True:
            try:
                try:
                    _create_unverified_https_context = ssl._create_unverified_context
                except AttributeError:
                    LOG.info('create unverified context failed')
                    pass
                else:
                    ssl._create_default_https_context = _create_unverified_https_context
                    LOG.info('create default https context success')
                jdata = json.dumps(body)
                if six.PY3:
                    jdata = jdata.encode('utf-8')
                req = Request(url, headers=headers, data=jdata)
                if context is not None:
                    response = urlopen(req, timeout=cst.TIME_SLEEP_60, context=context).read()
                else:
                    LOG.error("context is none, no need to verify!")
                    response = urlopen(req, timeout=cst.TIME_SLEEP_60).read()
                result = json.loads(response)
                return result
            except HTTPError as e:
                if conn_count > cst.CONNECT_COUNT:
                    login_msg = ('connect URL failed,error code:%s, error reason:%s, requestid:%s' % (
                        e.code, e.read(), e.headers.get('X-Request-Id')))
                    error_msg = convert_error(ErrorCode.BD_CONNECT_KMS_FAILED, 'connect URL failed')
                    LOG.error(login_msg)
                    raise Exception(error_msg)
                elif e.code in [cst.SERVER_TIMEOUT, cst.SERVICE_UNAVAILABLE, cst.INTERBAL_SERVER_ERROR]:
                    warn_msg = ('network has problem(httperror), try again later.error code:%s' % e.code)
                    LOG.info(warn_msg)
                    conn_count += 1
                    time.sleep(cst.TIME_SLEEP_30)
                    continue
                elif e.code == cst.BAD_REQUEST:
                    context_msg = ('bad request,error code:%s, error reason:%s' % (e.code, e.read()))
                    error_msg = convert_error(ErrorCode.BD_CONNECT_KMS_FAILED, context_msg)
                    LOG.error(context_msg)
                    raise Exception(error_msg)
                else:
                    login_msg = ('connect URL failed, error code:%s, error reason:%s, requestid:%s' % (
                        e.code, e.read(), e.headers.get('X-Request-Id')))
                    error_msg = convert_error(ErrorCode.BD_CONNECT_KMS_FAILED, login_msg)
                    LOG.error(login_msg)
                    raise Exception(error_msg)

            except URLError as e:
                if conn_count > cst.CONNECT_COUNT:
                    login_msg = ('connect URL failed, error reason:%s' % e.reason)
                    error_msg = convert_error(ErrorCode.BD_CONNECT_KMS_FAILED, 'connect URL failed')
                    LOG.error(login_msg)
                    raise Exception(error_msg)
                elif isinstance(e.reason, socket.timeout):
                    conn_count += 1
                    LOG.info('Connect URL failed, error reason is timeout, try again later.')
                    time.sleep(cst.TIME_SLEEP_30)
                    continue
                else:
                    conn_count += 1
                    LOG.info('network has problem(urlerror), try again later.')
                    time.sleep(cst.TIME_SLEEP_30)
                    continue

            except Exception as msg:
                LOG.error(msg)
                if conn_count > cst.CONNECT_COUNT:
                    login_msg = 'Connect URL failed, error reason is timeout.'
                    error_msg = convert_error(ErrorCode.BD_KMS_CERT_FILE_APTH_ILLEGAL, login_msg)
                    LOG.error(error_msg)
                    raise Exception(error_msg)
                else:
                    conn_count += 1
                    LOG.info('Connect URL is timeout, try again later.')
                    time.sleep(cst.TIME_SLEEP_30)
                    continue

    def put(self, url=None, headers=None, body=None):
        pass

    def delete(self, url=None, headers=None, body=None):
        pass


class KmsServerConf(object):
    def __init__(self):
        LOG.info("To get KMS ServerConf info...")
        self.server_url = self._get_server_url()
        self.server_ca_file = self._get_server_ca()

    def _get_server_url(self):
        server_url = CINDER_CONF.api_gateway_host
        if server_url != '' and len(server_url) < cst.LENGTH_LIMIT_255:
            return server_url
        else:
            msg = 'get server_url failed,server_url:%s' % server_url
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            LOG.info(error_msg)
            return ""

    def _get_server_ca(self):
        server_ca_file = CINDER_CONF.api_gateway_ca_file
        if os.path.exists(server_ca_file):
            return server_ca_file
        else:
            msg = 'get server_ca_file failed,server_ca_file:%s' % server_ca_file
            LOG.info(msg)
            return ""


class KMSRestClient(object):
    def __init__(self):
        LOG.info("kms init:")
        self.api_gateway_conf = KmsServerConf()

    def is_api_gateway_conf_correct(self):
        result = True
        if not self.api_gateway_conf.server_url:
            result = False
        return result

    def create_data_key(self, tenant_token, CMKID, project_id):  # interface,get VK and DEK
        LOG.info("connect to api-gateway to get VK and DEK")
        url = self.api_gateway_conf.server_url + "/v1.0/" + project_id + "/kms/create-datakey"
        create_datakey_body = {
            "key_id": CMKID,
            "datakey_length": str(cst.DATAKEY_LENGTH)
        }
        headers = {"Content-type": "application/json",
                   "X-Auth-Token": tenant_token
                   }
        self._verify_ca_pem()
        rest_con = RestConnection()
        result = rest_con.post(url, headers, create_datakey_body)
        VK = result.get("cipher_text")
        if VK is None:
            error_info = result.get('error', '')
            error_code = error_info.get('error_code', ErrorCode.BD_CONNECT_KMS_FAILED)
            error_desc = error_info.get('error_msg', "get VK failed")
            error_msg = convert_error(error_code, error_desc)
            LOG.error(error_msg)
            raise Exception.InvalidBackup(error_msg)
        LOG.info("get VK successful.")

        DEK = result.get("plain_text")
        if DEK is None:
            error_info = result.get('error', '')
            error_code = error_info.get('error_code', ErrorCode.BD_CONNECT_KMS_FAILED)
            error_desc = error_info.get('error_msg', "get DEK failed")
            error_msg = convert_error(error_code, error_desc)
            LOG.error(error_msg)
            raise Exception.InvalidBackup(error_msg)
        LOG.info("get DEK successful.")
        return VK, DEK

    def create_dec_key(self, token, cmk_id_alias, project_id):
        url = self.api_gateway_conf.server_url + "/v1.0/" + project_id + "/kms/create-key"
        self._verify_ca_pem()
        rest_con = RestConnection()
        body = {
            "key_alias": cmk_id_alias
        }
        headers = {"Content-type": "application/json",
                   "X-Auth-Token": token
                   }
        result = rest_con.post(url, headers, body)
        LOG.info("response dec key:%s", result)
        if result.get('error'):
            error_info = result.get('error', '')
            error_code = error_info.get('error_code', ErrorCode.BD_CONNECT_KMS_FAILED)
            error_desc = error_info.get('error_msg', "query key list failed")
            error_msg = convert_error(error_code, error_desc)
            raise Exception.InvalidBackup(error_msg)
        key_info = result.get("key_info")
        return key_info.get('key_id')

    def query_key(self, token, project_id):
        url = self.api_gateway_conf.server_url + "/v1.0/" + project_id + "/kms/list-keys"
        self._verify_ca_pem()
        rest_con = RestConnection()
        body = {"limit": "",
                "marker": ""
                }
        headers = {"Content-type": "application/json",
                   "X-Auth-Token": token
                   }
        result = rest_con.post(url, headers, body)
        LOG.info("result:%s", result)
        if result.get('error'):
            error_info = result.get('error', '')
            error_code = error_info.get('error_code', ErrorCode.BD_CONNECT_KMS_FAILED)
            error_desc = error_info.get('error_msg', "query key list failed")
            error_msg = convert_error(error_code, error_desc)
            raise Exception.InvalidBackup(error_msg)
        return result

    def decrypt_datakey(self, tenant_token, CMKID, project_id, VK):  # interface 12
        LOG.info("begin to get DEK")
        url = self.api_gateway_conf.server_url + "/v1.0/" + project_id + "/kms/decrypt-datakey"
        decrypt_datakey_body = {
            "key_id": CMKID,
            "datakey_cipher_length": cst.DATAKEY_CIPER_LENGTH,
            "cipher_text": VK
        }
        headers = {"Content-type": "application/json",
                   "X-Auth-Token": tenant_token
                   }
        self._verify_ca_pem()
        rest_con = RestConnection()
        result = rest_con.post(url, headers, decrypt_datakey_body)
        DEK = result.get("data_key")
        if DEK is None:
            error_info = result.get('error', '')
            error_code = error_info.get('error_code', ErrorCode.BD_CONNECT_KMS_FAILED)
            error_desc = error_info.get('error_msg', "get DEK failed")
            error_msg = convert_error(error_code, error_desc)
            LOG.error(error_msg)
            raise Exception.InvalidBackup(error_msg)
        LOG.info("get DEK successful.")
        return DEK

    def _verify_ca_pem(self):
        try:
            server_ca_file = self.api_gateway_conf.server_ca_file
            if not os.path.exists(server_ca_file):
                LOG.error('certificate error: ca file is not exist.')
                return
            else:
                url_part = self.api_gateway_conf.server_url.split("://")[1]
                if len(url_part.split(":")) == 1:
                    ssl.get_server_certificate(url_part, ssl_version=ssl.PROTOCOL_TLSv1_2,
                                               ca_certs=server_ca_file)
                elif len(url_part.split(":")) == 2:
                    IP = url_part.split(":")[0]
                    port = url_part.split(":")[1]
                    ssl.get_server_certificate((IP, int(port)), ssl_version=ssl.PROTOCOL_TLSv1_2,
                                               ca_certs=server_ca_file)
                else:
                    LOG.error('certificate error: server url is invalid.')
        except Exception as msg:
            LOG.warn('certificate error:' + str(msg))


class EVS(object):
    def __init__(self):
        LOG.info("try to init EVS ...")
        self.CMKID = ""
        self.VK = ""
        self.DEK = ""

    def set_encrypt_keys(self, db, context, backup):
        provider_service_str = backup.get(cst.PROVIDER_METADATA)
        if provider_service_str in (None, ''):
            provider_service = dict()
        else:
            provider_service = json.loads(provider_service_str)
        provider_service[cst.CMKID_KEY] = self.CMKID
        provider_service[cst.VK_KEY] = self.VK
        provider_service[cst.DEK_KEY] = Utils.encrypt_password(self.DEK)

        parameter_update = dict()
        parameter_update['provider_metadata'] = Utils.transfer_to_json_str(provider_service)
        call_db_fun(db.backup_update, context,
                    backup['id'],
                    parameter_update)

    def get_encrypt_keys(self, db, context, backup):
        backup_result = call_db_fun(db.backup_get, context, backup['id'])
        provider_metadata = backup_result.get(cst.PROVIDER_METADATA)
        if provider_metadata not in (None, ''):
            metadata = json.loads(provider_metadata)
            self.CMKID = metadata.get(cst.CMKID_KEY, '')
            self.VK = metadata.get(cst.VK_KEY, '')
            ENC_DEK = metadata.get(cst.DEK_KEY, '')
            if ENC_DEK:
                self.DEK = Utils.decrypt_password(ENC_DEK)

    def create_evs_body(self, db, context, backup, save):
        if not os.path.exists('/etc/huawei/dj/cfg/sys.ini'):
            volume_info = call_db_fun(db.volume_get, context, backup["volume_id"])
            if self._get_encrypted(volume_info):
                LOG.info("do encrypted backup")
                if save and cst.PROVIDER_METADATA in backup.keys():
                    self.get_encrypt_keys(db, context, backup)
                if self.CMKID == '' or self.VK == '' or self.DEK == '':
                    self.CMKID = self._get_volume_CMKID(volume_info)
                    is_csbs = False
                    vmid = Utils.transfer_json_to_dict(
                        backup.get("display_description")).get("VMID", "")
                    if is_uuid_like(vmid):
                        is_csbs = True
                    self.VK, self.DEK = self._get_VK_and_DEK(
                        db, context, backup['project_id'], backup["volume_id"],
                        self.CMKID, is_csbs=is_csbs)
                    # try to save vk dek to provider_metadata
                    if save and cst.PROVIDER_METADATA in backup.keys():
                        self.set_encrypt_keys(db, context, backup)
            else:
                LOG.info("do non-encrypt backup")
                self.CMKID = ""
                self.VK = ""
                self.DEK = ""
        body = {
            "DATA_ENC_INFO": {
                "CMKID": self.CMKID,
                "VK": self.VK,
                "DEK": self.DEK
            }
        }
        return body

    @staticmethod
    def get_replication_cmk_id(context, cmk_id_alias, project_id):
        kms_conn = KMSRestClient()
        if not kms_conn.is_api_gateway_conf_correct():
            msg = 'api gateway is not configed'
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            raise Exception(error_msg)
        key_list = kms_conn.query_key(context.auth_token, project_id)
        key_details = key_list.get('key_details', [])
        for key in key_details:
            if key.get('key_alias') != cmk_id_alias:
                continue
            return key.get('key_id')
        return kms_conn.create_dec_key(context.auth_token, cmk_id_alias, project_id)

    def get_remote_replication_body(self, db, context, cmk_id, project_id, volume_id):
        self.CMKID = cmk_id
        self.VK, self.DEK = self._get_VK_and_DEK(db, context, project_id, volume_id, self.CMKID)
        return self.VK, self.DEK

    def get_replication_body_if_backup_is_encrypt(self, db, context, project_id, volume_id):
        last_backup_item = self._get_last_CMKID_VK_backup(db, context, volume_id)
        if last_backup_item is not None:
            LOG.info("CMKID and VK exist in cinder DB, backup id:%s", last_backup_item['id'])
            metadata = Utils.transfer_json_to_dict(last_backup_item.get('service_metadata'))
            kms_conn = KMSRestClient()
            if not kms_conn.is_api_gateway_conf_correct():
                msg = 'api gateway is not configed'
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                raise Exception(error_msg)
            self.VK = metadata.get('VK')
            self.CMKID = metadata.get('CMKID')
            if self.VK and self.CMKID:
                self.DEK = kms_conn.decrypt_datakey(context.auth_token, self.CMKID, project_id, self.VK)
        else:
            LOG.info("this volume has not encrypt backup")
        return self.CMKID, self.VK, self.DEK

    def decrypt_evs_body(self, db, context, backup, save):
        if not os.path.exists('/etc/huawei/dj/cfg/sys.ini'):
            if cst.PROVIDER_METADATA in backup.keys():
                self.get_encrypt_keys(db, context, backup)
            if self.CMKID == '' or self.VK == '' or self.DEK == '':
                if backup['service_metadata'] not in (None, ''):
                    metadata = json.loads(backup['service_metadata'])
                    self.CMKID = metadata.get('CMKID', '')
                    self.VK = metadata.get('VK', '')
                    if self.CMKID != "" and self.VK != "":
                        LOG.info("do encrypt restore or delete")
                        kms_conn = KMSRestClient()
                        self.DEK = kms_conn.decrypt_datakey(context.auth_token, self.CMKID, backup['project_id'],
                                                            self.VK)
                        if save and cst.PROVIDER_METADATA in backup.keys():
                            self.set_encrypt_keys(db, context, backup)
                    elif self.CMKID == "" and self.VK != "":
                        msg = 'CMKID is Null and VK isnot Null'
                        error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                        LOG.error(error_msg)
                        raise Exception.InvalidBackup(error_msg)
                    elif self.CMKID != "" and self.VK == "":
                        msg = 'CMKID isnot Null and VK is Null'
                        error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                        LOG.error(error_msg)
                        raise Exception.InvalidBackup(error_msg)

        body = {
            "DATA_ENC_INFO": {
                "CMKID": self.CMKID,
                "VK": self.VK,
                "DEK": self.DEK
            }
        }
        return body

    def _get_encrypted(self, volume_info):
        encrypted = "0"
        volume_metadata = volume_info['volume_metadata']
        for metadata in volume_metadata:
            if metadata.key == '__system__encrypted':
                encrypted = metadata.value
                break
        if encrypted == "1":
            if KmsServerConf().server_url != "":
                LOG.info('The volume type is encrypted and kms server_url has configured.')
                return True
            else:
                LOG.info('The volume type is encrypted but kms server_url not configured.')
                return False
        elif encrypted == "0":
            LOG.info('The volume type is non-encrypted.')
            return False
        else:
            msg = '__system__encrypted is invalid,will do normal backup'
            LOG.error(msg)
            return False

    def _get_volume_CMKID(self, volume_info):
        volume_metadata = volume_info['volume_metadata']
        for metadata in volume_metadata:
            if metadata.key == '__system__cmkid':
                cmk_id = metadata.value
                if cmk_id != "":
                    return cmk_id
                break
        msg = 'invalid cmkid'
        LOG.error(msg)
        error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, "invalid cmkid")
        raise Exception(error_msg)

    def _get_VK_and_DEK(self, db, context, project_id, volume_id, CMKID,
                        is_csbs=True):
        LOG.info("get VK and DEK")
        kms_conn = KMSRestClient()
        last_backup_item = self._get_last_CMKID_VK_backup(
            db, context, volume_id, is_csbs=is_csbs)
        if last_backup_item is not None:
            LOG.info("CMKID and VK exist in cinder DB")
            metadata = json.loads(last_backup_item.get('service_metadata'))
            VK = metadata.get('VK')
            DEK = kms_conn.decrypt_datakey(context.auth_token, CMKID, project_id, VK)
        else:
            VK, DEK = kms_conn.create_data_key(context.auth_token, CMKID, project_id)
        return VK, DEK

    def _get_last_CMKID_VK_backup(self, db, context, volume_id, is_csbs=True):
        LOG.info("volume_id:%s" % volume_id)
        try:
            backup_list = self.__backup_get_by_volume_id(db, context, volume_id)
            tmp_latest_backup = None
            for backup_item in backup_list:
                try:
                    metadata = json.loads(backup_item.get('service_metadata'))
                except Exception:
                    LOG.error("Loads service metadata failed")
                    continue
                if 'CMKID' in metadata and 'VK' in metadata:
                    if metadata.get("CMKID") in [None, ''] or metadata.get("VK") in [None, '']:
                        continue
                    vmid = Utils.transfer_json_to_dict(
                        backup_item.get("display_description")).get("VMID", "")
                    if is_csbs:
                        if is_uuid_like(vmid):
                            if not tmp_latest_backup:
                                tmp_latest_backup = backup_item
                        else:
                            continue
                    else:
                        if is_uuid_like(vmid):
                            continue
                        else:
                            if not tmp_latest_backup:
                                tmp_latest_backup = backup_item
                    # get latest backup
                    if backup_item.get('created_at') > tmp_latest_backup.get('created_at'):
                        tmp_latest_backup = backup_item
            if tmp_latest_backup is None:
                LOG.info("backup of the volume does not has CMKID and VK in cinder DB")
                return None
            return tmp_latest_backup
        except Exception:
            LOG.exception("Get VK and CMKID from last backup failed.")
            return None

    def __backup_get_by_volume_id(self, db, context, volume_id):
        '''db interface is different for M and J versioin'''
        backup_list = []
        if hasattr(db, 'backup_get_by_volume_id'):
            backup_list = call_db_fun(db.backup_get_by_volume_id, context, volume_id)
        elif hasattr(db, 'backup_get_all_by_volume'):
            backup_list = call_db_fun(db.backup_get_all_by_volume, context, volume_id)
        else:
            LOG.error("db interface for get backup from volume can not be found.")
        return backup_list


# only for dj env
def _load_kmc():
    kmc = CDLL(_KMC_LIB_PATH)
    kmc.aesEncrypt.argtypes = [c_int, c_char_p]
    kmc.aesDecrypt.argtypes = [c_int, c_char_p]
    kmc.aesInit.argtypes = [c_char_p]
    kmc.aesInit.restype = c_int
    kmc.aesEncrypt.restype = c_char_p
    kmc.aesDecrypt.restype = c_char_p
    data_path = "/home/openstack/"
    kmc.aesInit(data_path)
    return kmc


fsc_cli = "fsc_cli"
CMD_BIN = fsc_cli + ' '


class FSPythonApi(object):
    def __init__(self, dsware_manager, fusionstorageagent):
        LOG.debug("FSPythonApi init")
        self.res_idx = len('result=')
        self.dsware_manager = dsware_manager
        self.fusionstorageagent = fusionstorageagent.split(',')
        self.create_log_file()

    def create_log_file(self):
        from cinder import utils
        try:
            from os import stat
            from pwd import getpwuid
            fsc_log_file = "/var/log/fsc_cli.log"
            if os.path.exists(fsc_log_file):
                owner_name = getpwuid(stat(fsc_log_file).st_uid).pw_name
                if owner_name == 'openstack':
                    return
        except Exception as e:
            LOG.warn("check fsc_cli file failed: %s" % e)

        try:
            fsc_cmd_touch = "sudo touch " + fsc_log_file
            fsc_cmd_chown = "sudo chown openstack:openstack " + fsc_log_file
            fsc_cmd_touch_end = tuple(fsc_cmd_touch.split())
            fsc_cmd_chown_end = tuple(fsc_cmd_chown.split())

            utils.execute(*fsc_cmd_touch_end)
            utils.execute(*fsc_cmd_chown_end)
        except Exception as pe:
            LOG.error("cmd execute failed, exception:%(ex)s" % {'ex': pe})

    def get_ip_port(self):
        return self.fusionstorageagent

    def get_manage_ip(self):
        return self.dsware_manager

    def start_execute_cmd(self, cmd, type):
        from cinder import utils
        fsc_ip = self.get_ip_port()
        manage_ip = self.get_manage_ip()
        ip_num = len(fsc_ip)

        LOG.info("fsc_ip is %s", fsc_ip)
        random.shuffle(fsc_ip)
        if ip_num <= 0:
            return None

        if ip_num > 3:
            ip_num = 3
            fsc_ip = fsc_ip[:ip_num]

        exec_result = ''
        result = ''

        if type:
            for ip in fsc_ip:
                cmd_args = CMD_BIN + cmd + ' --manage_ip ' + manage_ip.replace(
                    '\n', '') + ' --ip ' + ip.replace('\n', '')
                cmd_end = tuple(cmd_args.split())

                exec_result, err = utils.execute(*cmd_end, run_as_root=False)
                exec_result = exec_result.split('\n')
                LOG.info("result is %(result)s" % {'result': exec_result})
                if exec_result:
                    for line in exec_result:
                        if re.search('^result=0', line):
                            return exec_result
                        elif re.search('^result=50150007', line):
                            return 'result=0'
                        elif re.search('^result=5', line):
                            continue
            return exec_result
        else:
            for ip in fsc_ip:
                cmd_args = CMD_BIN + cmd + ' --manage_ip ' + manage_ip.replace(
                    '\n', '') + ' --ip ' + ip.replace('\n', '')
                cmd_end = tuple(cmd_args.split())

                exec_result, err = utils.execute(*cmd_end, run_as_root=False)
                exec_result = exec_result.split('\n')
                LOG.info("result is %(result)s" % {'result': exec_result})
                if exec_result:
                    for line in exec_result:
                        if re.search('^result=', line):
                            result = line
                            if re.search('^result=0', line):
                                return line
                            elif re.search('^result=50150007', line):
                                return 'result=0'
                            elif re.search('^result=5', line):
                                continue
            return result

    def create_LLD_volume(self, volume_name, pool_id, volume_size, isThin,
                          image_id, dataSrcUrl, image_size, offset, cacheFlag,
                          is_encrypted, cmkId, volume_auth_token, replace):

        cmd = '--op createLLDVolume' + ' ' + '--volName' + ' ' + str(
            volume_name) + ' ' + '--poolId' + ' ' + str(
            pool_id) + ' ' + '--volSize' + ' ' + str(
            volume_size) + ' ' + '--thinFlag' + ' ' + str(
            isThin) + ' ' + '--imageUUID' + ' ' + str(
            image_id) + ' ' + '--dataSrcURL' + ' ' + str(
            dataSrcUrl) + ' ' + '--imageSize' + ' ' + str(
            image_size) + ' ' + '--imageOffset' + ' ' + str(
            offset) + ' ' + '--cacheFlag' + ' ' + str(
            cacheFlag) + ' ' + '--replace' + ' ' + str(replace)

        backupData = json.loads(dataSrcUrl)
        if is_encrypted is not None and cmkId is not None and \
                        volume_auth_token is not None:
            cmd = cmd + ' ' + '--encrypted' + ' ' + str(
                is_encrypted) + ' ' + '--cmkId' + ' ' + str(
                cmkId) + ' ' + '--authCredentials' + ' ' + str(
                volume_auth_token)
        elif backupData.get('encrypt') == '1':  # if backup is encrypt, token is needed in FusionStorage
            cmd = cmd + ' ' + '--encrypted' + ' ' + str(0) + ' ' + '--cmkId' \
                  + ' ' + str(0) + ' ' + '--authCredentials' + ' ' + str(volume_auth_token)

        exec_result = self.start_execute_cmd(cmd, 0)
        if exec_result:
            if re.search('^result=0', exec_result):
                return 0
            else:
                return exec_result[self.res_idx:]
        else:
            return 1

    def snapshot_rollback(self, snap_name, volume_name, pool_id):
        cmd = '--op rollbackSnapshot --snapName {snap_name} ' \
              '--volName {volume_name} ' \
              '--poolId {pool_id}'.format(snap_name=snap_name,
                                          volume_name=volume_name,
                                          pool_id=pool_id)
        exec_result = self.start_execute_cmd(cmd, 0)
        if exec_result:
            if re.search('^result=0', exec_result):
                return 0
            else:
                return exec_result[self.res_idx:]
        else:
            return 1


class NovaApi(object):
    def __init__(self, nova_endpoint="", nova_ecs_api=""):
        self.tag_enable = bool(nova_endpoint and nova_ecs_api)
        self.nova_endpoint = nova_endpoint
        self.nova_ecs_api = nova_ecs_api
        self.nova_ecs_version = 1  # 1 represent /v1, 2 represent /v2.0
        if nova_ecs_api:
            self.nova_ecs_version = 2 if nova_ecs_api.find('v2') > 0 else 1

    def get_vm_tags(self, context, vm_id, project_id):
        if not self.tag_enable:
            LOG.info("ecs tag is disable because of nova endpoint or ecs api are not config")
            return {}

        vpc_id = self.get_server_info(context, vm_id, project_id)
        vm_tags = {}
        tags_api = {1: self.get_v1_tags, 2: self.get_v2_1_tags}
        func = tags_api.get(self.nova_ecs_version)

        result = func(context, vm_id, project_id, vm_tags, vpc_id)
        if result:
            return vm_tags

        return {}

    def get_server_info(self, context, vm_id, project_id):
        url = self.nova_endpoint + '/' + project_id + '/servers' + '/' + vm_id
        LOG.info("get vm detail: %s" % url)
        response = self.nova_get_api(context, url)
        server_info = response.get("server", {})
        metadata = server_info.get("metadata", None)
        vpc_id = ""
        if metadata:
            vpc_id = metadata.get("vpc_id", "")
        return vpc_id

    def nova_get_api(self, context, req_url):
        url = req_url
        headers = {"Content-Type": "application/json", "X-Auth-Token": context.auth_token}
        LOG.info("req vm tags url:%s", url)
        r = requests.get(url, headers=headers, verify=False)
        if r.status_code != 200:
            LOG.info("get vm tags failed, http status code:%d", r.status_code)
            return {}
        response = r.json()

        return response

    def get_v1_tags(self, context, vm_id, project_id, vm_tags, vpc_id):
        try:
            url = self.nova_ecs_api + '/' + project_id + '/servers/' + vm_id + '/tags'
            response = self.nova_get_api(context, url)
            if "tags" not in response:
                LOG.info("can't fetch v1 tags")
                return False
            if len(response.get('tags', [])) > 0:
                tags_list = response['tags']
                LOG.info("tags_list:%s", tags_list)
                for t in tags_list:
                    LOG.info("%s, %s", t['key'], t['value'])
                    if (t['key'] == vpc_id and t['value'] == "") or (
                                    t['key'] == '__type_baremetal' and t['value'] == ""):
                        continue
                    vm_tags[t['key']] = t['value']
            return True
        except Exception as e:
            LOG.info("get v1 tags failed:%s", e)
            return False

    def get_v2_1_tags(self, context, vm_id, project_id, vm_tags, vpc_id):
        try:
            url = self.nova_ecs_api + '/' + project_id + '/servers/' + vm_id + '/tags'
            response = self.nova_get_api(context, url)
            if "tags" not in response:
                LOG.info("can't fetch v2.1 tags")
                return False
            if len(response.get('tags', [])) > 0:
                tags_list = response['tags']
                LOG.info("tags_list:%s", tags_list)
                for tag in tags_list:
                    index = tag.find('=')
                    if index > 0:
                        LOG.info("vm tags contains equal sign")
                    else:
                        index = tag.find('.')
                    if index > 0:
                        LOG.info("%s, %s", tag[:index], tag[index + 1:])
                        if (tag[:index] == vpc_id and tag[index + 1:] == "") or \
                                (tag[:index] == '__type_baremetal' and tag[index + 1:] == ""):
                            continue
                        vm_tags[tag[:index]] = tag[index + 1:]
                    else:
                        LOG.info("%s", tag)
                        if tag == vpc_id or tag == '__type_baremetal':
                            continue
                        vm_tags[tag] = ""
            return True
        except Exception as e:
            LOG.info("get v2.1 tags failed:%s", e)
            return False
